<?php
// Prevent any output before headers
error_reporting(E_ALL);
ini_set('display_errors', 0);

session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once 'api_helpers.php';
require_once '../includes/functions.php';
require_once '../includes/sms_functions.php';
require_once '../includes/email_functions.php';
require_once '../includes/whatsapp_functions.php';
require_once '../includes/email_templates.php';

// Set headers for JSON response
header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    echo json_encode(['success' => false, 'message' => 'Invalid request data']);
    exit;
}

// Validate required fields
if (!isset($data['type']) || !isset($data['customer_id']) || !isset($data['message'])) {
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}

try {
    // Get store information
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'website') AND setting_group = 'store_info'");
    $stmt->execute();
    $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    $storeName = $storeSettings['store_name'] ?? 'Our Store';
    $storeAddress = $storeSettings['store_address'] ?? '';
    $storePhone = $storeSettings['store_phone'] ?? '';
    $storeEmail = $storeSettings['store_email'] ?? '';
    $storeWebsite = $storeSettings['website'] ?? '';

    // Get customer details
    $stmt = $pdo->prepare("SELECT name, phone, email FROM customers WHERE id = ?");
    $stmt->execute([$data['customer_id']]);
    $customer = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$customer) {
        echo json_encode(['success' => false, 'message' => 'Customer not found']);
        exit;
    }

    // Get customer's current balance and unpaid items (using same logic as sales receipt)
    $stmt = $pdo->prepare("
        SELECT 
            ct.id,
            ct.sale_id,
            s.invoice_number,
            s.created_at as sale_date,
            ct.amount,
            COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as credit_paid,
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
            ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
            COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
        WHERE ct.customer_id = ?
        GROUP BY ct.id
        HAVING balance > 0
        ORDER BY s.created_at DESC
    ");
    $stmt->execute([$data['customer_id']]);
    $unpaidItems = $stmt->fetchAll();
    
    // Get items and payment details for each unpaid invoice
    foreach ($unpaidItems as &$item) {
        // Get sale items
        $stmt = $pdo->prepare("
            SELECT p.name, si.quantity, si.price
            FROM sale_items si
            JOIN products p ON si.product_id = p.id
            WHERE si.sale_id = ?
            ORDER BY p.name
        ");
        $stmt->execute([$item['sale_id']]);
        $item['items'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get POS payment details (non-credit payments from original sale)
        $stmt = $pdo->prepare("
            SELECT method, amount 
            FROM sale_payments 
            WHERE sale_id = ? AND LOWER(method) != 'credit'
            ORDER BY id
        ");
        $stmt->execute([$item['sale_id']]);
        $pos_payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate POS paid amount
        $pos_paid_amount = 0;
        foreach ($pos_payments as $payment) {
            $pos_paid_amount += $payment['amount'];
        }
        
        // Add payment information to item
        $item['pos_payments'] = $pos_payments;
        $item['pos_paid_amount'] = $pos_paid_amount;
        $item['total_paid_amount'] = $pos_paid_amount + $item['credit_paid'];
        
        // Correct balance calculation: Credit Amount - Credit Payments - Waived
        // POS payments are already accounted for in the credit amount
        $item['balance'] = $item['amount'] - $item['credit_paid'] - $item['waived'];
        
        // Format the date for display
        $item['date'] = date('F d, Y', strtotime($item['sale_date']));
        $item['invoice'] = $item['invoice_number'];
    }
    
    // Filter out items that are now fully paid (balance <= 0)
    $unpaidItems = array_filter($unpaidItems, function($item) {
        return $item['balance'] > 0.001;
    });
    
    $totalBalance = array_sum(array_column($unpaidItems, 'balance'));

    $success = false;
    $error = '';
    $whatsappUrl = null;

    // Send based on type
    if ($data['type'] === 'sms' && !empty($customer['phone'])) {
        // Send SMS
        try {
            sendSMS($customer['phone'], $data['message']);
            $success = true;
        } catch (Exception $e) {
            $error = 'Failed to send SMS: ' . $e->getMessage();
        }
    } elseif ($data['type'] === 'email' && !empty($customer['email'])) {
        // Send Email with beautiful template
        try {
            $emailMessage = createPaymentReminderEmail(
                $customer['name'],
                $unpaidItems,
                $totalBalance,
                'Ksh', // You might want to get this from settings
                $storeName,
                $storeAddress,
                $storePhone,
                $storeEmail,
                $storeWebsite,
                $data['customer_id']
            );
            
            sendEmail($customer['email'], 'Payment Reminder - ' . $storeName, $emailMessage);
            $success = true;
        } catch (Exception $e) {
            $error = 'Failed to send email: ' . $e->getMessage();
        }
    } elseif ($data['type'] === 'whatsapp' && !empty($customer['phone'])) {
        // Send WhatsApp
        try {
            $result = sendWhatsApp($customer['phone'], $data['message']);
            if ($result['success']) {
                $success = true;
                // Store WhatsApp URL if available for link-based methods
                if (isset($result['whatsapp_url'])) {
                    $whatsappUrl = $result['whatsapp_url'];
                }
            } else {
                $error = 'Failed to send WhatsApp message: ' . ($result['message'] ?? 'Unknown error');
            }
        } catch (Exception $e) {
            $error = 'Failed to send WhatsApp message: ' . $e->getMessage();
        }
    } else {
        $error = 'Invalid reminder type or missing contact information';
    }

    // Log the reminder attempt
    $logDetails = json_encode([
        'customer_id' => $data['customer_id'],
        'customer_name' => $customer['name'],
        'type' => $data['type'],
        'success' => $success,
        'error' => $error
    ]);
    
    $stmt = $pdo->prepare("
        INSERT INTO system_logs (user_id, action, details) 
        VALUES (?, 'credit_reminder', ?)
    ");
    $stmt->execute([$_SESSION['user_id'], $logDetails]);

    $response = [
        'success' => $success,
        'message' => $success ? 'Reminder sent successfully' : $error
    ];

    // Include WhatsApp URL if available
    if ($whatsappUrl) {
        $response['whatsapp_url'] = $whatsappUrl;
        $response['requires_manual_send'] = true;
        $response['message'] = 'WhatsApp link generated successfully. Click to open WhatsApp and send the message.';
    }

    echo json_encode($response);

} catch (Exception $e) {
    error_log("Reminder error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while sending the reminder'
    ]);
} 